﻿
/****************************************************************************/
/*Copyright (c) 2011, Florent DEVILLE.                                      */
/*All rights reserved.                                                      */
/*                                                                          */
/*Redistribution and use in source and binary forms, with or without        */
/*modification, are permitted provided that the following conditions        */
/*are met:                                                                  */
/*                                                                          */
/* - Redistributions of source code must retain the above copyright         */
/*notice, this list of conditions and the following disclaimer.             */
/* - Redistributions in binary form must reproduce the above                */
/*copyright notice, this list of conditions and the following               */
/*disclaimer in the documentation and/or other materials provided           */
/*with the distribution.                                                    */
/* - The names of its contributors cannot be used to endorse or promote     */
/*products derived from this software without specific prior written        */
/*permission.                                                               */
/* - The source code cannot be used for commercial purposes without         */
/*its contributors' permission.                                             */
/*                                                                          */
/*THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       */
/*"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT         */
/*LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS         */
/*FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE            */
/*COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,       */
/*INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,      */
/*BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;          */
/*LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER          */
/*CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT        */
/*LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN         */
/*ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE           */
/*POSSIBILITY OF SUCH DAMAGE.                                               */
/****************************************************************************/

using System.Collections.Generic;

using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Content;

//using GE.Gui.Forms;
using GE.Visualisation;
using GE.Message;
using GE.Input;

namespace GE.Gui
{
    /// <summary>
    /// GUI Component.
    /// </summary>
    class Gui
    {
        /// <summary>
        /// Pause between 2 Gui inputs.
        /// </summary>
        private static int INPUT_PAUSE = 250;

        /// <summary>
        /// Unique instance of the Gui class.
        /// </summary>
        private static Gui m_instance = new Gui ();

        /// <summary>
        /// Maps the assets to graphic IDs.
        /// </summary>
        private Dictionary<string, int> m_resources;

        /// <summary>
        /// Stack of windows.
        /// </summary>
        private LinkedList<Window> m_windowStack;

        /// <summary>
        /// Current active window.
        /// </summary>
        private Window m_activeWindow;

        /// <summary>
        /// Stack of input modes.
        /// </summary>
        private LinkedList<InputMode> m_modeStack;

        private LinkedList<Window> m_hideQueue;

        private ContentManager m_content;

        /// <summary>
        /// Unique instance of the Gui class.
        /// </summary>
        public static Gui Instance
        {
            get { return m_instance; }
        }

        public ContentManager Content
        {
            get { return m_content; }
            set { m_content = value; }
        }

        /// <summary>
        /// Private constructor.
        /// </summary>
        private Gui()
        {
            m_resources = new Dictionary<string, int>();
            m_windowStack = new LinkedList<Window>();
            m_hideQueue = new LinkedList<Window>();
            m_activeWindow = null;

            m_modeStack = new LinkedList<InputMode>();
        }

        /// <summary>
        /// Load a texture
        /// </summary>
        /// <param name="assetName">asset name</param>
        /// <returns>graphic ID</returns>
        public int loadTexture(string assetName)
        {
            //try
            //{
            //    return m_resources[assetName];
            //}
            //catch (KeyNotFoundException)
            //{
                return Visu.Instance.loadTexture(assetName);
            //}
        }

        /// <summary>
        /// Show a window.
        /// </summary>
        /// <param name="window">window to show</param>
        public void showWindow(Window window)
        {
            m_activeWindow = window;
            m_windowStack.AddLast(window);
            
            m_modeStack.AddFirst(Input.Input.Instance.Mode);
            Input.Input.Instance.Mode = InputMode.Gui;
        }

        /// <summary>
        /// Hide (close) a window.
        /// Actually add it to the hide queue.
        /// </summary>
        /// <param name="window">window to hide</param>
        public void hideWindow(Window window)
        {
            m_hideQueue.AddLast(window);
        }

        /// <summary>
        /// Hide (close) a window.
        /// </summary>
        /// <param name="window">window to hide</param>
        private void hideWindowImpl(Window window)
        {
            m_windowStack.Remove(window);
            if (m_windowStack.Count > 0)
                m_activeWindow = m_windowStack.Last.Value;
            else
                m_activeWindow = null;

            Input.Input.Instance.Mode = m_modeStack.First.Value;
            m_modeStack.RemoveFirst();
        }

        /// <summary>
        /// Hide all windows.
        /// </summary>
        public void hideAllWindows()
        {
            m_windowStack.Clear();
            m_activeWindow = null;
            Input.Input.Instance.Mode = m_modeStack.Last.Value;
            m_modeStack.Clear();
        }

        /// <summary>
        /// Stop inputs for a while.
        /// </summary>
        public void pauseInput()
        {
            Input.Input.Instance.pause(INPUT_PAUSE);
        }

        /// <summary>
        /// Update the Gui component.
        /// Sends events to windows.
        /// </summary>
        public void update()
        {
            if (m_activeWindow == null)
                return;

            while (m_hideQueue.Count > 0)
            {
                hideWindowImpl(m_hideQueue.First.Value);
                m_hideQueue.RemoveFirst();
            }

            foreach(Message.Message m in MessageSystem.Instance.getList(MessageReceiver.Gui))
            {
                if (m.Type == MessageType.Input)
                {
                    /////////////////////////////////////////////////////////////////////////////////////////
                    // To make with MessageType.Gui then map it !
                    /////////////////////////////////////////////////////////////////////////////////////////
                    if ((GameAction)m.A == GameAction.Validate)
                        m_activeWindow.sendEvent(WindowEvent.Validate);
                    if ((GameAction)m.A == GameAction.Cancel)
                        m_activeWindow.sendEvent(WindowEvent.Cancel);
                    else if ((GameAction)m.A == GameAction.Left)
                        m_activeWindow.sendEvent(WindowEvent.Left);
                    else if ((GameAction)m.A == GameAction.Right)
                        m_activeWindow.sendEvent(WindowEvent.Right);
                    else if ((GameAction)m.A == GameAction.Up)
                        m_activeWindow.sendEvent(WindowEvent.Up);
                    else if ((GameAction)m.A == GameAction.Down)
                        m_activeWindow.sendEvent(WindowEvent.Down);
                    Message.MessageSystem.Instance.popMessage(m.Index);
                    /////////////////////////////////////////////////////////////////////////////////////////
                }
            }

            foreach (Window w in m_windowStack)
                w.update();
        }

        /// <summary>
        /// Render the Gui component.
        /// </summary>
        public void render()
        {
            foreach (Window w in m_windowStack)
                w.render();
        }
    }
}
